# subprocess_tool.py
import subprocess
import json
import os

TOOL_RUNNER_PATH = os.path.abspath("/root/IQA/IQA-Agent/AgenticIQA/qwen/tool_runner.py")  
PYIQA_ENV_NAME = "pyiqa_env"
CUDA_DEVICE = "3"
def execute_tool_by_name(tool_name: str, **kwargs) -> float:
    args_str = json.dumps(kwargs)

    try:
        result = subprocess.run(
            ["conda", "run", "-n", "pyiqa_env", "python", TOOL_RUNNER_PATH, tool_name, args_str],
            capture_output=True,
            text=True,
            check=True
        )
        stdout_lines = result.stdout.strip().splitlines()
        stderr = result.stderr.strip()

        if not stdout_lines:
            raise RuntimeError(f"[Tool Error] Empty stdout. stderr: {stderr}")

        json_line = stdout_lines[-1]

        try:
            output = json.loads(json_line)
        except json.JSONDecodeError as je:
            raise RuntimeError(f"[Tool Error] Invalid JSON Output:\nstdout: {result.stdout}\nstderr: {stderr}")

        if "error" in output:
            raise RuntimeError(f"[Tool Error] {output['error']}")

        return output["score"]

    except subprocess.CalledProcessError as e:
        raise RuntimeError(f"[Subprocess Failed] {e.stderr}")

import asyncio
import time
import traceback
from typing import Dict, Any

class ToolExecutor:
    def __init__(self, max_concurrent_tasks: int = 1):
        self.semaphore = asyncio.Semaphore(max_concurrent_tasks)

    async def run(self, tool_name: str, args: Dict[str, Any]) -> float:
        async with self.semaphore:
            try:
                start = time.time()
                score = await asyncio.to_thread(self._execute_subprocess, tool_name, args)
                print(f"[ToolExecutor] {tool_name} done on cuda:{CUDA_DEVICE} in {time.time() - start:.2f}s")
                return score
            except Exception as e:
                print(f"[ToolExecutor] Failed: {tool_name} with error: {e}")
                traceback.print_exc()
                raise

    def _execute_subprocess(self, tool_name: str, args: Dict[str, Any]) -> float:
        args["device"] = "cuda"
        args_str = json.dumps(args)

        env = os.environ.copy()
        env["CUDA_VISIBLE_DEVICES"] = CUDA_DEVICE

        try:
            result = subprocess.run(
                ["conda", "run", "-n", PYIQA_ENV_NAME, "python", TOOL_RUNNER_PATH, tool_name, args_str],
                capture_output=True,
                text=True,
                check=True,
                env=env
            )
            stdout_lines = result.stdout.strip().splitlines()
            stderr = result.stderr.strip()

            if not stdout_lines:
                raise RuntimeError(f"[Tool Error] Empty stdout. stderr: {stderr}")

            json_line = stdout_lines[-1]

            try:
                output = json.loads(json_line)
            except json.JSONDecodeError:
                raise RuntimeError(f"[Tool Error] Invalid JSON Output:\nstdout: {result.stdout}\nstderr: {stderr}")

            if "error" in output:
                raise RuntimeError(f"[Tool Error] {output['error']}")
            return output["score"]

        except subprocess.CalledProcessError as e:
            raise RuntimeError(f"[Subprocess Failed] {e.stderr}")

tool_executor = ToolExecutor(max_concurrent_tasks=1)
